import { Component } from "../../api/gui/Component.js"
import { watchResize } from "../../lib/type/element/watchResize.js"
import { keyboard } from "../../api/env/device/keyboard.js"

import * as THREE from "../../../c/libs/threejs/0.181/three.js"

// import { OrbitControls } from "../../../c/libs/threejs/0.181/addons/controls/OrbitControls.js"
// import { ArcballControls } from "../../../c/libs/threejs/0.181/addons/controls/ArcballControls.js"
import { ArcballControls } from "./model/ArcballControls.js"
import { getBasename } from "../../lib/syntax/path/getBasename.js"
import { Loader } from "../../lib/graphic/webgl/ModelLoader.js"
import { defer } from "../../lib/type/promise/defer.js"
import { isInstanceOf } from "../../lib/type/any/isInstanceOf.js"

// TODO: check https://github.com/gkjohnson/three-jumpflood-demo
// TODO: check https://github.com/gkjohnson/three-mesh-bvh

// @src https://github.com/tiesfa/threejs_autoscaler/blob/main/js/3dPreviewer.js
function autoScale(mesh) {
  // This part auto scales the model
  const box = new THREE.Box3().setFromObject(mesh)

  const width = box.max.x - box.min.x
  const height = box.max.y - box.min.y

  const scaleFactor =
    width >= height
      ? 2 / width // Scale based on the width (2)
      : 1.5 / height // Scale based on the height (1.5)

  mesh.scale.setScalar(scaleFactor)

  // Adjust model position on Y axis
  box.setFromObject(mesh)

  const newModelHeight = box.max.y - box.min.y

  const heightAdjust = newModelHeight / 2 - box.max.y
  mesh.position.set(0, heightAdjust, 0)
}

export class ModelComponent extends Component {
  static plan = {
    tag: "ui-model",
    props: {
      src: true,
      visualization: true,
    },
  }

  get src() {
    return this.getAttribute("src")
  }
  set src(value) {
    this.setAttribute("src", value)
  }

  get autoplay() {
    return this.hasAttribute("autoplay")
  }
  set autoplay(value) {
    this.toggleAttribute("autoplay", value)
  }

  get visualization() {
    return this.getAttribute("visualization") ?? "solid"
  }
  set visualization(value) {
    this.setAttribute("visualization", value)
  }

  #solid = new Set()
  setWireframe() {
    if (!this.model) return

    const v = this.visualization

    const solid = v === "both" || v === "solid"

    if (solid) {
      const toRemove = []
      this.model?.traverse((obj) => {
        if (this.#solid.has(obj.uuid)) obj.visible = true
        if (obj.name === "wireframe") toRemove.push(obj)
      })

      if (v === "solid") {
        for (const item of toRemove) item.removeFromParent()
        return
      }
    }

    this.model.traverseVisible((obj) => {
      // console.log(obj)
      if (obj.type === "Mesh" || obj.type === "SkinnedMesh") {
        obj.visible = solid
        this.#solid.add(obj.uuid)
        const linesGeometry = new THREE.EdgesGeometry(obj.geometry)
        const lines = new THREE.LineSegments(
          linesGeometry,
          new THREE.LineBasicMaterial({
            color: "#000000",
          }),
        )

        lines.name = "wireframe"

        // console.log(lines)

        lines.position.x = obj.position.x
        lines.position.y = obj.position.y
        lines.position.z = obj.position.z
        lines.rotation.x = obj.rotation.x
        lines.rotation.y = obj.rotation.y
        lines.rotation.z = obj.rotation.z

        obj.parent.add(lines)
      }
    })
  }

  constructed() {
    
    const renderer = new THREE.WebGLRenderer({
      alpha: true,
      autoClear: false,
      preserveDrawingBuffer: true,
      stencil: true,
      antialias: false,
      autoClearDepth: false,
      autoClearStencil: false,
    })
    renderer.autoClearColor = false

    const scene = new THREE.Scene()
    const camera = new THREE.PerspectiveCamera(45, 1, 0.1, 1000)
    camera.position.set(0, 0, 3)

    const hemiLight = new THREE.HemisphereLight(0xcc_cc_cc, 0x44_44_44, 3)
    scene.add(hemiLight)

    const ambLight = new THREE.AmbientLight(0x40_40_40)
    scene.add(ambLight)

    const dirLight = new THREE.DirectionalLight(0xff_ff_ff, 2.5)
    dirLight.position.set(1.5, 3, 2.5)
    scene.add(dirLight)

    const group = new THREE.Group()
    scene.add(group)

    this.mode = true

    const controls = new ArcballControls(camera, renderer.domElement, scene)
    controls.setGizmosVisible(false)
    controls.dampingFactor = 0
    controls.enableDamping = true
    controls.enablePan = true
    controls.enableRotate = !this.mode 
    controls.update()

    this.clearEnabled = true;
    let spacePrev = false;

    keyboard.listen()
    const keySpeedSlow = 0.006
    const keySpeedMid = 0.02
    const keySpeedFast = 0.08
    let keySpeed = keySpeedMid

    const zoomOptions = { deltaX: 0, ctrlKey: false, metaKey: false, shiftKey: false, preventDefault: () => { } }
    function zoomUp() { zoomOptions.deltaY = -500 * keySpeed; controls._onWheel(zoomOptions) }
    function zoomDown() { zoomOptions.deltaY = 500 * keySpeed; controls._onWheel(zoomOptions) }

    this.clock = new THREE.Clock()

    let rotationVelocity = new THREE.Vector2()
    const rotationDamping = 0
    let isLeftMouseDown = false
    let lastMouseX = 0
    let lastMouseY = 0

    renderer.domElement.addEventListener("mousedown", (e) => {
      if (this.mode && e.button === 0) {
        isLeftMouseDown = true
        lastMouseX = e.clientX
        lastMouseY = e.clientY
      }
    })
    renderer.domElement.addEventListener("mouseup", (e) => { if (this.mode && e.button === 0) isLeftMouseDown = false })
    renderer.domElement.addEventListener("mousemove", (e) => {
      if (!this.mode || !isLeftMouseDown) return
      const deltaX = e.clientX - lastMouseX
      const deltaY = e.clientY - lastMouseY
      lastMouseX = e.clientX
      lastMouseY = e.clientY
      const rotationSpeed = 0.005
      group.rotation.y += deltaX * rotationSpeed
      group.rotation.x += deltaY * rotationSpeed
      rotationVelocity.set(deltaX * rotationSpeed, deltaY * rotationSpeed)
    })

    this.loop = () => {
      const delta = this.clock.getDelta()

      for (const anim of this.animations) anim.update(delta)

      if (this.mode && !isLeftMouseDown) {
        group.rotation.y += rotationVelocity.x
        group.rotation.x += rotationVelocity.y
        rotationVelocity.multiplyScalar(1 - rotationDamping)
        if (rotationVelocity.length() < 0.0001) rotationVelocity.set(0, 0)
      }

      keySpeed = keyboard.codes.ShiftLeft ? keySpeedFast : keyboard.codes.ControlLeft ? keySpeedSlow : keySpeedMid
      if (keyboard.codes.ShiftRight) zoomUp()
      else if (keyboard.codes.ControlRight) zoomDown()

      if (keyboard.codes.ArrowLeft) group.rotation.y -= keySpeed
      else if (keyboard.codes.ArrowRight) group.rotation.y += keySpeed
      if (keyboard.codes.ArrowUp) group.rotation.x -= keySpeed
      else if (keyboard.codes.ArrowDown) group.rotation.x += keySpeed

      if (keyboard.codes.Space && !spacePrev) {
        this.clearEnabled = !this.clearEnabled; 
      }
      //if (!keyboard.codes.Space) renderer.clear()
      spacePrev = keyboard.codes.Space; // toggle
      if (this.clearEnabled) renderer.clear();
      renderer.render(scene, camera)

    }

    this.renderer = renderer
    this.camera = camera
    this.scene = scene
    this.group = group
    this.animations = []
    this.loader = new Loader()
    this.modelReady = defer()

  }

  addCube() {
    const mesh = new THREE.Mesh(
      new THREE.BoxGeometry(1, 1, 1),
      new THREE.MeshPhongMaterial({ color: 0x00_ff_00 }),
    )

    const wireframe = new THREE.LineSegments(
      new THREE.EdgesGeometry(mesh.geometry), // or WireframeGeometry
      new THREE.LineBasicMaterial({ color: 0x00_00_00 }),
    )

    const group = new THREE.Group()
    // group.add(mesh)
    group.add(wireframe)
    this.scene.add(group)
  }

  async updated(key, val) {
    switch (key) {
      case "src": {
        await this.load(val)
        break
      }

      case "visualization": {
        this.setWireframe()
        break
      }

      default:
        break
    }
  }

  async #loadModel(urlOrBlob, path) {
    let blob

    if (isInstanceOf(urlOrBlob, Blob)) {
      blob = urlOrBlob
    } else {
      blob = await (await fetch(urlOrBlob)).blob()
      blob.name ??= getBasename(urlOrBlob)
      path ??= urlOrBlob
    }

    return { blob, blobPath: path }
  }

  async load(urlOrBlob, path) {
    this.modelReady ??= defer()

    const { blob, blobPath } = await this.#loadModel(urlOrBlob, path)
    const model = await this.loader.loadFile(blob, blobPath)

    autoScale(model)
    // console.log(model)

    if (model.animations?.length) {
      const { animations } = model
      const mixer = new THREE.AnimationMixer(model)
      mixer.clipAction(animations[0]).play()
      this.animations.push(mixer)
    }

    this.group.clear()
    this.group.add(model)
    this.model = model

    // const helper = new THREE.SkeletonHelper(model)
    // this.group.add(helper)

    this.modelReady.resolve()
    this.modelReady = undefined

    this.setWireframe()
  }

  async loadAnimation(urlOrBlob, path) {
    await this.modelReady

    let model
    if (urlOrBlob) {
      const { blob, blobPath } = await this.#loadModel(urlOrBlob, path)
      model = await this.loader.loadFile(blob, blobPath)
    }

    for (const item of this.animations) item.stopAllAction()
    this.animations.length = 0

    if (model?.animations?.length) {
      const { animations } = model
      const mixer = new THREE.AnimationMixer(this.model)
      mixer.clipAction(animations[0]).play()
      this.animations.push(mixer)
    }
  }

  render() {
    if (this.src) this.updated("src", this.src)
    return this.renderer.domElement
  }

  created() {
    const { signal } = this
    watchResize(this, { signal, firstCall: true }, ({ width, height }) => {
      this.camera.aspect = width / height
      this.camera.updateProjectionMatrix()
      this.renderer.setSize(width, height)
      this.loop()
    })

    this.renderer.setAnimationLoop(this.loop)
  }
}

export const model = Component.define(ModelComponent)
