import { Component } from "../../api/gui/Component.js"
import { createFuzzySearch, sortByScore } from "../../lib/algo/fuzzySearch.js"
import { fuzzyResultsToPlan } from "../../api/gui/helper/fuzzyResultsToPlan.js"
import { Frecency } from "../../lib/structure/Frecency.js"
import { dispatch } from "../../lib/event/dispatch.js"
import { MenuComponent } from "./menu.js"
import { noop } from "../../lib/type/function/noop.js"
import { i18n } from "../../api/i18n.js"

/**
 * @import { FuzzyResult } from "../../lib/algo/fuzzySearch.js"
 *
 * @typedef {{
 *   prefix?: string,
 *   load: (() => any[] | Promise<any[]>),
 *   getText?: (item: any) => string,
 *   frecency?: boolean,
 * }} ListboxMode
 *
 * @typedef {{
 *   text: string,
 *   item?: any,
 *   score?: number,
 *   matches?: FuzzyResult["matches"],
 * }} ListboxEntry
 */

// MARK: Presets
// =============

/** @type {Record<string, () => ListboxMode>} */
const presets = {
  apps: () => ({
    prefix: ">",
    async load() {
      const { os } = await import("../../api/os.js")
      await os.apps.ready
      return Object.values(os.apps.value).map(({ name }) => name)
    },
  }),
  exec: () => ({
    prefix: "$",
    async load() {
      const { os } = await import("../../api/os.js")
      await os.apps.ready
      return Object.values(os.apps.value).map(({ command }) => command)
    },
  }),
  help: () => ({
    prefix: "?",
    frecency: false,
    async load() {
      const { inApple } = await import("../../api/env/browser/inApple.js")
      return [
        { label: "… Open file", shortcut: inApple ? "⌘+P" : "Ctrl+P" },
        { label: "> Apps", shortcut: inApple ? "⌘+Shift+P" : "Ctrl+Shift+P" },
        { label: "? Help" },
      ]
    },
  }),
  files: () => ({
    async load() {
      const { os } = await import("../../api/os.js")
      const homeItems = []
      const items = []
      os.fileIndex.readDir("/", { recursive: true, absolute: true }, (text) => {
        if (
          text.endsWith("/") ||
          text.endsWith(".desktop") ||
          text.endsWith(".directory")
        ) {
          return
        }

        if (text.startsWith(os.env.HOME)) {
          homeItems.push(text)
        } else {
          items.push(text)
        }
      })
      return [...homeItems, ...items]
    },
  }),
}

/**
 * @param {any} def
 * @param {string} name
 * @returns {ListboxMode | undefined}
 */
function resolveMode(def, name) {
  if (def === true) return presets[name]?.()
  if (typeof def === "object") {
    const preset = presets[name]?.()
    return preset ? { ...preset, ...def } : def
  }
}

// MARK: getText
// =============

function defaultGetText(item) {
  if (typeof item === "string") return item
  return item?.label ?? item?.text ?? item?.value ?? ""
}

/**
 * @param {any[]} source
 * @param {(item: any) => string} [getText]
 * @returns {ListboxEntry[]}
 */
function toEntries(source, getText = defaultGetText) {
  const out = []
  for (const item of source) {
    const text = getText(item)
    out.push(typeof item === "string" ? { text } : { text, item })
  }
  return out
}

/**
 * @param {ListboxEntry} entry
 * @returns {any}
 */
function entryToPlan(entry) {
  const sourceItem = entry.item

  // Spread source plan properties (label, picto, shortcut, disabled, etc.)
  const plan =
    sourceItem && typeof sourceItem === "object" ? { ...sourceItem } : {}

  if (entry.matches) {
    plan.label = () => fuzzyResultsToPlan(entry)
  } else {
    plan.label ??= entry.text
  }

  return plan
}

// MARK: Listbox
// =============

export class ListboxComponent extends MenuComponent {
  static plan = {
    tag: "ui-listbox",
    role: "listbox",
    id: true,
  }

  /** Items after filtering/sorting, before conversion to plans. */
  items = []

  emptyLabel = i18n("No matching results")

  itemRole = "option"
  fuzzy = true
  frecency = false
  limit = 50

  /** @type {(item: any) => string} */
  getText = noop

  /** @type {(item: any) => any} */
  renderElement = noop

  #contentRaw
  #source = []
  #entries = []
  #defaultResults = []
  #fuzzySearchFn
  #frecencyInstance
  #search = ""
  #oldmode
  #mode
  #loadedMode
  #modes = {}
  #history = {}
  #needsResolve = true

  // MARK: search
  get search() {
    return this.#search
  }
  set search(query) {
    this.#search = query
    if (this.isRendered) this.rerender()
  }

  // MARK: history
  get history() {
    return this.#history
  }
  set history(value) {
    this.#history = value ?? {}
    if (this.#frecencyInstance) {
      this.#frecencyInstance.history = this.#history
    }
  }

  // MARK: mode
  get mode() {
    return this.#mode
  }
  set mode(name) {
    this.#oldmode = this.#mode
    this.#mode = name
    this.#loadedMode = undefined
    if (this.isRendered) this.rerender()
  }

  // MARK: modes
  get modes() {
    return this.#modes
  }
  set modes(value) {
    this.#modes = value ?? {}
  }

  // MARK: highlight
  highlight(idx, currentItem) {
    const prev = this.ariaActiveDescendantElement
    const result = super.highlight(idx, currentItem)
    if (prev) prev.ariaSelected = null
    if (this.ariaActiveDescendantElement) {
      this.ariaActiveDescendantElement.ariaSelected = "true"
    }
    return result
  }

  // MARK: activate
  activate(idx, e, target) {
    if (target?.disabled || target?.dataset.disabled === "true") return
    this.items[idx]?.action?.(e, target)
    this.pick(idx)
  }

  // MARK: pick
  pick(idx) {
    const entry = this.items[idx]
    if (!entry) return

    if (this.frecency && this.#frecencyInstance) {
      if (this.#search) {
        this.#frecencyInstance.recordSelection("", entry.text)
      }
      this.#frecencyInstance.recordSelection(this.#search, entry.text)
    }

    const detail = entry.item ?? entry
    dispatch(this, "ui:listbox.pick", { detail })
  }

  // MARK: #buildIndex
  #buildIndex() {
    const getText = this.getText === noop ? defaultGetText : this.getText
    this.#entries = toEntries(this.#source, getText)
    this.#defaultResults = this.#entries
      .map((e) => ({ ...e, score: 0 }))
      .sort(sortByScore)

    this.#fuzzySearchFn =
      this.fuzzy && this.#entries.length > 0
        ? createFuzzySearch(this.#entries.map((e) => e.text))
        : undefined

    if (this.frecency && !this.#frecencyInstance) {
      this.#frecencyInstance = new Frecency({
        accessor: (entry) => entry.text,
        // @ts-ignore
        history: this.#history,
      })
    }
  }

  async initMode(query = this.#search) {
    if (this.#modes) {
      for (const [name, def] of Object.entries(this.#modes)) {
        const mode = resolveMode(def, name)
        if (mode?.prefix && query.startsWith(mode.prefix)) {
          if (this.#loadedMode !== name) {
            await this.#loadMode(name)
          }

          return query.slice(mode.prefix.length).trim()
        }
      }

      // No prefix matched — reset to default if we were in a prefix mode
      if (
        this.#mode &&
        this.#loadedMode &&
        resolveMode(this.#modes[this.#mode], this.#mode)?.prefix
      ) {
        this.#mode = undefined
      }

      this.#mode ??= "files"

      if (this.#loadedMode !== this.#mode) {
        await this.#loadMode(this.#mode)
      }
    }

    return query
  }

  // MARK: render
  async render() {
    const query = await this.initMode()

    let list

    list =
      query && this.fuzzy && this.#fuzzySearchFn
        ? this.#fuzzySearchFn(query)
        : this.#defaultResults.map((e) => ({ ...e }))

    // Frecency sorting
    const modeConf = this.#mode
      ? resolveMode(this.#modes[this.#mode], this.#mode)
      : undefined
    const useFrecency =
      modeConf?.frecency !== false && this.frecency && this.#frecencyInstance

    if (useFrecency) {
      this.#frecencyInstance.sortResults(query ?? "", list)
    }

    // Limit
    if (this.limit < list.length) {
      list = list.slice(0, this.limit)
    }

    this._content = list
    return super.render().then((res) => {
      this.highlight(0)
      return res
    })
  }

  renderItem(item, i) {
    if (this.renderElement) {
      return super.renderItem(this.renderElement(item), i)
    }
    return super.renderItem(entryToPlan(item), i)
  }

  // MARK: #loadMode
  async #loadMode(name) {
    const oldmode = resolveMode(this.#modes[this.#oldmode], this.#oldmode)
    const mode = resolveMode(this.#modes[name], name)
    if (!mode) return

    this.#mode = name
    dispatch(this, "ui:listbox.mode-change", {
      detail: {
        mode,
        oldmode,
      },
    })

    let items
    if (typeof mode.load === "function") {
      items = await mode.load()
    } else if (Array.isArray(mode.load)) {
      items = mode.load
    }

    this.#loadedMode = name

    this.#source = items ?? []
    this.#buildIndex()
    return items
  }
}

export const listbox = Component.define(ListboxComponent)
