/* eslint-disable no-multi-assign */
//! Copyright (c) 2019 halvves. MIT License.
// @src https://github.com/halvves/shaderpen

// @read https://shadertoyunofficial.wordpress.com/2016/07/22/compatibility-issues-in-shadertoy-webglsl/

import { configure } from "../../../api/configure.js"
import { isMobile } from "../../../api/env/isMobile.js"
import { Program } from "./Program.js"

const SHADER_TOY_VERTEX = `\
#version 300 es

in highp vec2 position;

void main() {
  gl_Position = vec4(position, 0.0, 1.0);
}`
const SHADER_TOY_HEADER = `\
#version 300 es

precision highp float;
precision highp int;

#define HW_PERFORMANCE ${isMobile() ? "0" : "1"}

out vec4 fragColor;\n`

const SHADER_TOY_FOOTER = `\
void main() {
  mainImage(fragColor, gl_FragCoord.xy);
}`

const wrappers = {
  shaderToy: {
    vertex: SHADER_TOY_VERTEX,
    header: SHADER_TOY_HEADER,
    footer: SHADER_TOY_FOOTER,
    uniforms: {
      iResolution: {
        type: "vec3",
        value: [0, 0, globalThis.devicePixelRatio || 1],
      },
      iMouse: { type: "vec4" },
      iFrame: { type: "int" },
      iTime: { type: "float" },
      iTimeDelta: { type: "float" },
    },
  },
}

export class Shader {
  autoplay = false

  constructor(gl, source, options) {
    const { canvas } = gl
    this.canvas = canvas

    this.gl = gl

    gl.clearColor(0, 0, 0, 0)
    gl.pixelStorei(gl.UNPACK_FLIP_Y_WEBGL, true) // https://jameshfisher.com/2020/10/22/why-is-my-webgl-texture-upside-down/

    // create a 2D quad Vertex Buffer
    this.vertices = new Float32Array([-1, -1, 1, -1, -1, 1, 1, -1, 1, 1, -1, 1])

    const buffer = (this.buffer = gl.createBuffer())
    gl.bindBuffer(gl.ARRAY_BUFFER, buffer)
    gl.bufferData(gl.ARRAY_BUFFER, this.vertices, gl.STATIC_DRAW)

    this.uniforms = {}

    this.lastTime = 0

    this.configure(options)

    if (source) this.compile(source, options)
  }

  configure(options = {}) {
    options.uniforms ??= {}
    this.config = configure(
      options?.wrapper in wrappers
        ? wrappers[options.wrapper]
        : wrappers.shaderToy,
      options,
    )
  }

  compile(source, options) {
    const { gl } = this

    if (options) this.configure(options)

    let { header, footer, vertex } = this.config

    const uniforms = {}

    const setUniformMethod = (key, type, value) => {
      let method

      if (type.startsWith("vec")) {
        const n = type.at(-1)
        method = `uniform${n}fv`
        value ??= new Array(Number(n)).fill(0)
      } else {
        method = `uniform1${type[0]}`
        value ??= 0
      }

      if (this.uniforms[key]) value = this.uniforms[key].value

      uniforms[key] = { value, method }
    }

    const declaredUniforms = []
    source.replaceAll(
      /^uniform (?<type>\w+) (?<key>\w+);/gm,
      (_, type, key) => {
        declaredUniforms.push(key)
        setUniformMethod(key, type)
      },
    )

    for (const [key, val] of Object.entries(this.config.uniforms)) {
      if (declaredUniforms.includes(key)) {
        uniforms[key].value = val.value ?? val
      } else {
        const { type, value } = val
        header += `uniform ${type} ${key};\n`
        setUniformMethod(key, type, value)
      }
    }

    this.source = `${header}\n${source}\n${footer}`

    const program = new Program(gl, {
      fragment: this.source,
      vertex,
    })

    program.link()

    // keep playing previous program if an error is thrown
    program.use()

    this.program?.destroy()
    this.program = program
    this.uniforms = uniforms
    this.bindUniforms()
    this.setSize()

    if (this.paused === false) return

    if (this.config.autoplay) this.play()
    else this.render()
  }

  bindUniforms() {
    const { gl } = this
    const { position } = this.program.attribs

    gl.enableVertexAttribArray(position)
    gl.vertexAttribPointer(position, 2, gl.FLOAT, false, 0, 0)

    this.updates = []

    for (const key of Object.keys(this.program.uniforms)) {
      const location = this.program.uniforms[key]
      const uniform = this.uniforms[key]
      this.updates.push(() => gl[uniform.method](location, uniform.value))
    }
  }

  setSize(width = this.canvas.width, height = this.canvas.height) {
    this.uniforms.iResolution.value[0] = width
    this.uniforms.iResolution.value[1] = height
    this.gl.viewport(0, 0, width, height)
    this.render()
  }

  paused = true
  loop(time) {
    this.render(time)
    this.rafId = requestAnimationFrame((time) => this.loop(time))
  }

  play() {
    this.paused = false
    this.loop(0)
  }

  pause() {
    this.paused = true
    cancelAnimationFrame(this.rafId)
  }

  togglePause(force = !this.paused) {
    if (force) this.pause()
    else this.play()
  }

  clear() {
    this.gl.clear(this.gl.COLOR_BUFFER_BIT)
  }

  render(timestamp = this.lastTime) {
    const { gl } = this

    const delta = (timestamp - this.lastTime) / 1000
    this.lastTime = timestamp

    this.uniforms.iTime.value += delta
    this.uniforms.iTimeDelta.value = delta
    this.uniforms.iFrame.value++

    gl.clear(gl.COLOR_BUFFER_BIT)
    for (const update of this.updates) update(delta)
    gl.drawArrays(gl.TRIANGLES, 0, 6)
  }
}
